/**
 * Filter support
 */

(function() {

    if ('undefined' === typeof FILTER) {
        FILTER = {};
    }

    // Add missing regex literal escape method
    RegExp.escape = function(text) {
        return text.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, "\\$&");
    }

    // PRIVATE MEMBERS
    var _state = {
        search_terms: null,
        filters: {}
    };
    var _filters = null;
    var _products = null;

    // PRIVATE METHODS

    function _attach_handlers() {
        // Filter handlers
        $(".filter-block__list").on("click", "a", _on_filter_click);
        $(".filter-block__list").on("click", ".filter__title", _on_filter_title_click);

        // Search handler
        $(".filter-search").on("keyup", ".filter-search__terms", _on_search_keyup);
        $(".filter-search").on("click", ".filter-search__indicator", _on_search_clear);
        $(".filter-search").on("submit", function(event) {
            event.preventDefault();
        });

    }

    /**
     * Reacts when a filter value is clicked
     */
    function _on_filter_click(event) {
        event.preventDefault();

        var $target = $(event.currentTarget);
        var $filter = $target.closest(".filter");
        var filter_type = $filter.attr("data-filter-type");
        var filter_name = $filter.attr("data-filter-name");

        var filter_value = "";
        if ($target.attr("data-filter-value")) {
            filter_value = $target.attr("data-filter-value").toLowerCase();
        } else {
            filter_value = $target.text().toLowerCase();
        }

        var location = $.inArray(filter_value, _state.filters[filter_name]);

        if (filter_type === "multiselect") {
            if (location < 0) {
                _state.filters[filter_name].push(filter_value);
                $target.addClass("is-active");
            } else {
                _state.filters[filter_name].splice(location, 1);
                $target.removeClass("is-active");
            }
        } else if (filter_type === "radio") {
            if (location < 0) {
                _state.filters[filter_name] = [filter_value];
                $filter.find("a").removeClass("is-active");
                $target.addClass("is-active");
            } else {
                _state.filters[filter_name] = [];
                $target.removeClass("is-active");
            }
        } else {
            throw "Unknown filter type: " + filter_type;
        }

        _render();

    }

    function _on_filter_title_click(event) {
        event.preventDefault();

        var $target = $(event.currentTarget);
        var $filter = $target.closest(".filter");
        $filter.toggleClass("is-opened");
    }

    function _on_search_keyup(event) {
        _change_search_icon();
        _state.search_terms = $(event.currentTarget).val();
        _render();
    }

    function _on_search_clear(event) {
        $(".filter-search__terms").val("").trigger("keyup");
        _change_search_icon();
    }

    function _change_search_icon() {
        var terms = $(".filter-search__terms").val();
        var $form = $(".filter-search");
        var $indicator = $(".filter-search__indicator");

        if (terms === "") {
            $form.removeClass("has-search");
            $indicator.attr("data-icon", "g");
        } else {
            $form.addClass("has-search");
            $indicator.attr("data-icon", "x");
        }
    }

    function _render() {
        var $wall = $(".wall");
        var $existing_bricks = $wall.find(".brick-product");
        var $bricks_to_remove = $();
        var $bricks_to_insert = $();

        var visible_products = _get_filtered_products(_state.filters, _products);
        visible_products = _get_searched_products(_state.search_terms, visible_products);

        var visible_product_ids = $.map(visible_products, function(product, index) {
            return product.id;
        });

        $.each($existing_bricks, function(index, brick) {
            var $brick = $(brick);
            var id = parseInt($brick.attr("data-product-id"))
            var location = $.inArray(id, visible_product_ids);

            if (location < 0) {
                $bricks_to_remove = $bricks_to_remove.add($brick);
            } else {
                $bricks_to_insert = $bricks_to_insert.add($brick);
            }
        });

        $bricks_to_remove.addClass("brick--is-filtered");
        $bricks_to_insert.removeClass("brick--is-filtered");

        if (typeof BRICK !== "undefined") {
            BRICK.render();
        }

    }

    function _get_filtered_products() {
        var result = [];

        // For each product, check if ALL filters matche (i.e. and for each filter if at-least a value matches)
        $.each(_products, function(p_index, product) {
            var show_product = true;

            $.each(_filters, function(f_index, filter) {
                var match_exists = _filter_matches(filter, product);

                /* Exclusive rule ?, whereas inclusive */
                if (!match_exists) {
                    show_product = false;
                    return false; // break
                }
            });

            if (show_product) {
                result.push(product);
            }
        });

        return result;
    }

    function _filter_matches(filter, product) {
        // Match if no value selected for the filter
        if (_state.filters[filter].length === 0) {
            return true;
        }

        // Failed match if the filter is missing for the product (no value available)
        if (!(filter in product.filterables)) {
            return false;
        }

        // Otherwise check if at least one selected filter appears in the filter values for the product
        var match_exists = false;

        if ($.isArray(product.filterables[filter])) {
            $.each(product.filterables[filter], function(fa_index, filterable) {
                var location = $.inArray(filterable, _state.filters[filter]);
                if (location >= 0) {
                    match_exists = true;
                    return false; // break
                }
            });
        } else {
            var location = $.inArray(product.filterables[filter], _state.filters[filter]);
            match_exists = location >= 0;
        }

        return match_exists;
    }

    function _get_searched_products(terms, products) {
        if (terms === null || terms === "") {
            return products;
        }

        var result = [];
        var pattern = new RegExp(RegExp.escape(terms), "ig");

        $.each(products, function(p_index, product) {
            if (product.title.match(pattern) !== null || product.vendor.match(pattern) !== null) {
                result.push(product);
            }
        });

        return result;
    }

    function _transform_products() {
        var $bricks = $(".brick-product");
        var products = [];

        $.each($bricks, function(index, brick) {
            var $brick = $(brick);
            //var product = $.parseJSON($brick.attr("data-product"));
            var product = [];
            product.id = parseInt($brick.attr("data-product-id"));
            product.title = $brick.attr("data-product-title");
            product.categories = $brick.attr("data-product-categories");
            product.vendor = ''; //$brick.attr("data-product-vendor");
            products.push(product);
        });

        $.each(products, function(p_index, product) {
            var filterables = {};

            /*
            // Size and Gender
            var size_index = $.inArray("Size", product.options);
            if (size_index >= 0) {
                filterables.sizes = [];

                $.each(product.variants, function(v_index, variant) {
                    var size = variant.options[size_index].toLowerCase();

                    var has_infinite = (variant.inventory_management === null || variant.inventory_management === "");
                    var has_stock = (variant.inventory_management === "shopify" && variant.inventory_quantity > 0);

                    if ($.inArray(size, filterables.sizes) < 0 && (has_infinite || has_stock)) {
                        filterables.sizes.push(size);
                    }
                });
            }
*/

            /*
            // Availability
            var variant_statuses = $.map(product.variants, function(variant, index) {
                if (variant.inventory_management === null || variant.inventory_management === "") {
                    return "pre-order";
                }

                if (variant.inventory_management === "shopify") {
                    if (variant.inventory_quantity <= 0) {
                        return "sold out";
                    } else {
                        return "in-stock";
                    }
                }

                throw "Unknown inventory_management: " + variant.inventory_management;
            });

            filterables.availability = variant_statuses.reduce(function(accumulator, item) {
                if (accumulator === "pre-order" || item === "pre-order") {
                    return "pre-order";
                }

                if (accumulator === "in-stock" || item === "in-stock") {
                    return "in-stock";
                }

                if (accumulator === "sold out" && item === "sold out") {
                    return "sold out";
                }

                throw "Invalid inventory_management: " + accumulator + " and " + item;
            });
*/

            // Category filter
            filterables.category = [];
            var categories = product.categories.split(",");

            $.each(categories, function(index, category) {
                filterables.category.push(category.toLowerCase());
            });

            // Label filter : same values as Category filter (i.e. as labels values are categories)
            filterables.label = [];
            $.each(categories, function(index, category) {
                filterables.label.push(category.toLowerCase());
            });

            product.filterables = filterables;
        });

        return products;
    }

    function _create_filters(products) {
        var filters = [];

        $.each(products, function(p_index, product) {
            $.each(product.filterables, function(f_index, filterable) {
                if ($.inArray(f_index, filters) < 0) {
                    filters.push(f_index);
                }
            });
        });

        /** ACO hack for testing - i.e. with special filter-name */
        if ($.inArray('tst-category', filters) < 0) {
            filters.push('tst-category');
        }
        if ($.inArray('tst-affiner', filters) < 0) {
            filters.push('tst-affiner');
        }

        return filters;
    }

    function _initFilterStates() {

        /** ACO synchronise _state.filters avec les éléments actifs */
        var $filters = $('.filter');
        $.each($filters, function(index, filter) {
            var $filter = $(filter);
            var filter_type = $filter.attr("data-filter-type");
            var filter_name = $filter.attr("data-filter-name");
            var $activeValues = $('.is-active', $filter);
            if (filter_type === "multiselect") {

            } else if (filter_type === "radio") {
                if ($activeValues.length > 1) {
                    $activeValues = $activeValues.first();
                }
            } else {
                throw "Unknown filter type: " + filter_type;
            }
            $.each($activeValues, function(i, val) {
                var $target = $(val);
                var filter_value = "";
                if ($target.attr("data-filter-value")) {
                    filter_value = $target.attr("data-filter-value").toLowerCase();
                } else {
                    filter_value = $target.text().toLowerCase();
                }
                _state.filters[filter_name].push(filter_value);
            });
        });
        /** ACO synchronize _state.search_terms avec les éléments actifs */
        _state.search_terms = $(".filter-search__terms").val();
    }

    function _initFilterUI() {
        /** Activate filter values */
        var $activeValues = $(".filter .is-active");
        $.each($activeValues, function(i, val) {
            var $value = $(val);
            var $filter = $value.closest(".filter");
            $filter.addClass("is-opened");
        });
        /* Adapt icon */
        _change_search_icon();
    }

    // PUBLIC MEMBERS

    // PUBLIC METHODS
    FILTER.init = function() {
        // check that filter is available 
        var $filter = $(".filter");
        if ($filter.length == 0)
            return;
        console.log('FILTER.init start');
        _products = _transform_products();
        _filters = _create_filters(_products);
        $.each(_filters, function(index, filter) {
            _state.filters[filter] = [];
        });
        _initFilterStates();
        _initFilterUI();
        _attach_handlers();
        _render();
        console.log('FILTER.init end');
    };

})();